<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ClientController extends Controller
{
    use ApiResponse;

    public function index()
    {
        $clients = Client::with(['location', 'country'])
            ->latest()
            ->paginate(10);

        return $this->success(
            $clients,
            'Client list retrieved successfully'
        );
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'company' => 'required|string|max:255',
            'abbreviation' => 'nullable|string|max:50',

            'contact_first_name' => 'nullable|string|max:100',
            'contact_last_name' => 'nullable|string|max:100',
            'email' => 'nullable|email|max:255',
            'contact_date' => 'nullable|date',

            'website' => 'nullable|url',

            'main_phone' => 'nullable|string|max:30',
            'mobile_phone' => 'nullable|string|max:30',
            'fax' => 'nullable|string|max:30',

            'location_id' => 'nullable|exists:locations,id',
            'country_id' => 'nullable|exists:countries,id',

            'address_1' => 'nullable|string|max:255',
            'address_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip' => 'nullable|string|max:20',

            'cc_confirmations_to' => 'nullable|string|max:255',

            'shared_notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->error(
                $validator->errors(),
                'Validation failed',
                422
            );
        }

        $client = Client::create($validator->validated());

        return $this->success(
            $client,
            'Client created successfully',
            201
        );
    }

    public function show($id)
    {
        $client = Client::with(['location', 'country'])->find($id);

        if (!$client) {
            return $this->error(
                [],
                'Client not found',
                404
            );
        }

        return $this->success(
            $client,
            'Client retrieved successfully'
        );
    }

    public function update(Request $request, $id)
    {
        $client = Client::find($id);

        if (!$client) {
            return $this->error(
                [],
                'Client not found',
                404
            );
        }

        $validator = Validator::make($request->all(), [
            'company' => 'sometimes|required|string|max:255',
            'abbreviation' => 'nullable|string|max:50',

            'contact_first_name' => 'nullable|string|max:100',
            'contact_last_name' => 'nullable|string|max:100',
            'email' => 'nullable|email|max:255',
            'contact_date' => 'nullable|date',

            'website' => 'nullable|url',

            'main_phone' => 'nullable|string|max:30',
            'mobile_phone' => 'nullable|string|max:30',
            'fax' => 'nullable|string|max:30',

            'location_id' => 'nullable|exists:locations,id',
            'country_id' => 'nullable|exists:countries,id',

            'address_1' => 'nullable|string|max:255',
            'address_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip' => 'nullable|string|max:20',

            'cc_confirmations_to' => 'nullable|string|max:255',

            'shared_notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->error(
                $validator->errors(),
                'Validation failed',
                422
            );
        }

        $client->update($validator->validated());

        return $this->success(
            $client,
            'Client updated successfully'
        );
    }


    public function destroy($id)
    {
        $client = Client::find($id);

        if (!$client) {
            return $this->error(
                [],
                'Client not found',
                404
            );
        }

        $client->delete();

        return $this->success(
            [],
            'Client deleted successfully'
        );
    }
}
