<?php

namespace App\Http\Controllers\API;

use App\Models\Course;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;

class CourseController extends Controller
{
    use ApiResponse;

    public function index()
    {
        $data = Course::with(['addons', 'deposits'])->get();

        return response()->json(['success' => true, 'data' => $data], 200);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'course_name' => 'required|string|max:255',
            'mode' => 'required|integer',
            'discipline' => 'required|integer',
            'allow_deposit' => 'required|numeric|min:0',
            'prompt' => 'required|string',
            'shipping_price' => 'required|numeric|min:0',

            'keycode_bank_id' => 'nullable|exists:keycode_banks,id',
            'card_type_id' => 'nullable|exists:card_types,id',
            'second_card_type_id' => 'nullable|exists:second_card_types,id',

            'calendar_icon_color' => 'nullable|string|max:255',
            'ecu_credits' => 'nullable|string|max:255',
            'description' => 'nullable|string',

            'custom_sidebar' => 'nullable|boolean',

            'confirmation_email' => 'nullable|email|max:255',
            'email_subject' => 'nullable|string|max:255',
            'payment_confirmation_email' => 'nullable|email|max:255',
            'email_body' => 'nullable|string',

            'seo_rich_results' => 'nullable|integer|in:0,1',
            'seo_description' => 'nullable|integer|in:0,1',
            'selected_addons' => 'nullable|array',
            'deposit_amounts' => 'nullable|array',
            'multiple_price' => 'nullable|array',
            'multiple_price.*.price_level_prompt' => 'required|string',
            'multiple_price.*.price' => 'required|numeric',
            'multiple_price.*.code' => 'required|string',
        ]);

        $data = new Course();
        $data->course_name = $validated['course_name'];
        $data->mode = $validated['mode'];
        $data->discipline = $validated['discipline'];
        $data->allow_deposit = $validated['allow_deposit'];
        $data->prompt = $validated['prompt'];
        $data->shipping_price = $validated['shipping_price'];

        $data->keycode_bank_id = $validated['keycode_bank_id'] ?? null;
        $data->card_type_id = $validated['card_type_id'] ?? null;
        $data->second_card_type_id = $validated['second_card_type_id'] ?? null;

        $data->calendar_icon_color = $validated['calendar_icon_color'] ?? null;
        $data->ecu_credits = $validated['ecu_credits'] ?? null;
        $data->description = $validated['description'] ?? null;

        $data->custom_sidebar = $validated['custom_sidebar'] ?? false;

        $data->confirmation_email = $validated['confirmation_email'] ?? null;
        $data->email_subject = $validated['email_subject'] ?? null;
        $data->payment_confirmation_email = $validated['payment_confirmation_email'] ?? null;
        $data->email_body = $validated['email_body'] ?? null;

        $data->seo_rich_results = $validated['seo_rich_results'] ?? 0;
        $data->seo_description = $validated['seo_description'] ?? 0;

        $data->save();

        if (!empty($validated['selected_addons'])) {
            foreach ($validated['selected_addons'] as $addonId) {
                DB::table('selected_course_addons')->insert([
                    'course_id' => $data->id,
                    'addon_id' => $addonId,
                ]);
            }
        }
        if (!empty($request->multiple_price)) {
            foreach ($request->multiple_price as $priceData) {
                DB::table('deposit_amounts')->insert([
                    'course_id' => $data->id,
                    'price_level_prompt' => $priceData['price_level_prompt'],
                    'price_levels' => json_encode([
                        'price' => $priceData['price'],
                        'code' => $priceData['code'],
                        'description' => $priceData['description'] ?? null,
                    ]),
                ]);
            }
        }

        return $this->success($data, 'Data stored successfully!', 200);
    }

    public function update(Request $request)
    {
        $data = Course::find($request->id);

        if (!$data) {
            return $this->error(null, 'Data not found', 404);
        }

        $validated = $request->validate([
            'course_name' => 'required|string|max:255',
            'mode' => 'required|integer',
            'discipline' => 'required|integer',
            'allow_deposit' => 'required|numeric|min:0',
            'prompt' => 'required|string',
            'shipping_price' => 'required|numeric|min:0',

            'keycode_bank_id' => 'nullable|exists:keycode_banks,id',
            'card_type_id' => 'nullable|exists:card_types,id',
            'second_card_type_id' => 'nullable|exists:second_card_types,id',

            'calendar_icon_color' => 'nullable|string|max:255',
            'ecu_credits' => 'nullable|string|max:255',
            'description' => 'nullable|string',

            'custom_sidebar' => 'nullable|boolean',

            'confirmation_email' => 'nullable|email|max:255',
            'email_subject' => 'nullable|string|max:255',
            'payment_confirmation_email' => 'nullable|email|max:255',
            'email_body' => 'nullable|string',

            'seo_rich_results' => 'nullable|integer|in:0,1',
            'seo_description' => 'nullable|integer|in:0,1',
            'selected_addons' => 'nullable|array',
            'deposit_amounts' => 'nullable|array',
        ]);

        $data->course_name = $validated['course_name'];
        $data->mode = $validated['mode'];
        $data->discipline = $validated['discipline'];
        $data->allow_deposit = $validated['allow_deposit'];
        $data->prompt = $validated['prompt'];
        $data->shipping_price = $validated['shipping_price'];

        $data->keycode_bank_id = $validated['keycode_bank_id'] ?? null;
        $data->card_type_id = $validated['card_type_id'] ?? null;
        $data->second_card_type_id = $validated['second_card_type_id'] ?? null;

        $data->calendar_icon_color = $validated['calendar_icon_color'] ?? null;
        $data->ecu_credits = $validated['ecu_credits'] ?? null;
        $data->description = $validated['description'] ?? null;

        $data->custom_sidebar = $validated['custom_sidebar'] ?? false;

        $data->confirmation_email = $validated['confirmation_email'] ?? null;
        $data->email_subject = $validated['email_subject'] ?? null;
        $data->payment_confirmation_email = $validated['payment_confirmation_email'] ?? null;
        $data->email_body = $validated['email_body'] ?? null;

        $data->seo_rich_results = $validated['seo_rich_results'] ?? 0;
        $data->seo_description = $validated['seo_description'] ?? 0;

        $data->save();

        if (!empty($validated['selected_addons'])) {
            foreach ($validated['selected_addons'] as $addonId) {
                DB::table('selected_course_addons')->insert([
                    'course_id' => $data->id,
                    'addon_id' => $addonId,
                ]);
            }
        }

        if (!empty($validated['deposit_amounts'])) {
            foreach ($validated['deposit_amounts'] as $deposit) {
                DB::table('deposit_amounts')->insert([
                    'course_id' => $data->id,
                    'price_level_prompt' => $deposit['price_level_prompt'],
                    'price_levels' => json_encode([
                        'price' => $deposit['price'],
                        'code' => $deposit['code'],
                        'description' => $deposit['description'] ?? null,
                    ]),
                ]);
            }
        }

        return $this->success($data, 'Data update successfully!', 200);
    }

    public function delete(Request $request)
    {
        $data = Course::find($request->id);

        if (!$data) {
            return $this->error(null, 'Data not found', 404);
        }

        $data->delete();
        return $this->success($data, 'Data deleted successfully!', 200);
    }
}
