<?php

namespace App\Http\Controllers\Api;

use App\Models\Instructor;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class InstructorController extends Controller
{
    use ApiResponse;

    // Get All Instructors
    public function index()
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $instructors = Instructor::with('roles')->get();
        return $this->success($instructors, "Instructors Retrieved Successfully", 200);
    }

    // Get Single Instructor
    public function show($id)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $instructor = Instructor::with('roles')->find($id);

        if (!$instructor) {
            return $this->error([], "Instructor Not Found", 404);
        }

        return $this->success($instructor, "Instructor Retrieved Successfully", 200);
    }

    // Create Instructor
    public function store(Request $request)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $validator = Validator::make($request->all(), [
            'username' => 'required|string|unique:instructors,username',
            'training_site_id' => 'required|integer',

            'first_name' => 'required|string',
            'middle_name' => 'nullable|string',
            'last_name' => 'required|string',

            'address_line_1' => 'required|string',
            'address_line_2' => 'nullable|string',
            'city' => 'required|string',
            'state_province_region' => 'required|string',
            'zip_postal_code' => 'required|string',
            'country_id' => 'required|integer',

            'mobile_phone' => 'nullable|string',

            'name_to_print_on_card' => 'nullable|string',
            'aha_instructor_id' => 'nullable|string',
            'hsi_instructor_id' => 'nullable|string',
            'rclc_username' => 'nullable|string',

            'email' => 'required|email|unique:instructors,email',
            'password' => 'required|string|min:6',

            'active_user' => 'nullable|boolean',
            'read_only_user' => 'nullable|boolean',
            'allow_bid_on_open_classes' => 'nullable|boolean',

            'roles' => 'nullable|array',
            'roles.*' => 'integer|exists:roles,id',
        ]);

        $validated = $validator->validated();

        if ($validator->fails()) {
            return $this->error($validator->errors(), $validator->errors()->first(), 422);
        }

        try {
            $instructor = Instructor::create([
                'username' => $request->username,
                'training_site_id' => $request->training_site_id,

                'first_name' => $request->first_name,
                'middle_name' => $request->middle_name,
                'last_name' => $request->last_name,

                'address_line_1' => $request->address_line_1,
                'address_line_2' => $request->address_line_2,
                'city' => $request->city,
                'state_province_region' => $request->state_province_region,
                'zip_postal_code' => $request->zip_postal_code,
                'country_id' => $request->country_id,

                'mobile_phone' => $request->mobile_phone,

                'name_to_print_on_card' => $request->name_to_print_on_card,
                'aha_instructor_id' => $request->aha_instructor_id,
                'hsi_instructor_id' => $request->hsi_instructor_id,
                'rclc_username' => $request->rclc_username,

                'email' => $request->email,
                'password' => Hash::make($request->password),

                'active_user' => $request->active_user ?? true,
                'read_only_user' => $request->read_only_user ?? false,
                'allow_bid_on_open_classes' => $request->allow_bid_on_open_classes ?? false,
            ]);
            
            // Assign roles to newly created instructor
            if (!empty($validated['roles'])) {
                $pivotData = [];
                foreach ($validated['roles'] as $roleId) {
                    $pivotData[] = [
                        'instructor_id' => $instructor->id,
                        'role_id' => $roleId,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }
                DB::table('user_role')->insert($pivotData);
            }

            // Load roles for response
            $instructor->load('roles');

            return $this->success($instructor, "Instructor Created Successfully", 201);

        }catch (\Exception $e) {
            return $this->error([], $e->getMessage(), 500);
        }
    }

    // Update Instructor
    public function update(Request $request, $id)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $instructor = Instructor::find($id);
        if (!$instructor) {
            return $this->error([], "Instructor Not Found", 404);
        }

        // Validation
        $validator = Validator::make($request->all(), [
            'username' => 'nullable|string',
            'training_site_id' => 'nullable|integer',

            'first_name' => 'nullable|string',
            'middle_name' => 'nullable|string',
            'last_name' => 'nullable|string',

            'address_line_1' => 'nullable|string',
            'address_line_2' => 'nullable|string',
            'city' => 'nullable|string',
            'state_province_region' => 'nullable|string',
            'zip_postal_code' => 'nullable|string',
            'country_id' => 'nullable|integer',
            'mobile_phone' => 'nullable|string',

            'name_to_print_on_card' => 'nullable|string',
            'aha_instructor_id' => 'nullable|string',
            'hsi_instructor_id' => 'nullable|string',
            'rclc_username' => 'nullable|string',

            'email' => 'nullable|email',
            'password' => 'nullable|string|min:6',

            'active_user' => 'nullable|boolean',
            'read_only_user' => 'nullable|boolean',
            'allow_bid_on_open_classes' => 'nullable|boolean',

            'roles' => 'nullable|array',
            'roles.*' => 'integer|exists:roles,id',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors(), $validator->errors()->first(), 422);
        }

        $validated = $validator->validated();

        try {
            $updateData = $validated;

            if (!empty($validated['password'])) {
                $updateData['password'] = Hash::make($validated['password']);
            } else {
                unset($updateData['password']);
            }

            $roles = [];
            if (isset($updateData['roles'])) {
                $roles = $updateData['roles'];
                unset($updateData['roles']);
            }

            $instructor->update($updateData);

            if (!empty($roles)) {
                $instructor->roles()->sync($roles);
            }
            
            $instructor->load('roles');

            return $this->success($instructor, "Instructor Updated Successfully", 200);

        } catch (\Exception $e) {
            return $this->error([], $e->getMessage(), 500);
        }
    }

    // Delete Instructor
    public function destroy($id)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $instructor = Instructor::find($id);
        if (!$instructor) {
            return $this->error([], "Instructor Not Found", 404);
        }

        $instructor->delete();
        return $this->success([], "Instructor Deleted Successfully", 200);
    }
}
