<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Location;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LocationController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of locations
     */
    public function index()
    {
        $locations = Location::latest()->paginate(10);

        return $this->success($locations, 'Location list retrieved successfully');
    }

    /**
     * Store a newly created location
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'abbreviation' => 'nullable|string|max:50',
            'directions' => 'nullable|string',

            'print_card_line_1' => 'nullable|string|max:255',
            'print_card_line_2' => 'nullable|string|max:255',

            'address_1' => 'nullable|string|max:255',
            'address_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',

            'contact_name' => 'nullable|string|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:30',

            'internal_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->error(
                $validator->errors(),
                'Validation failed',
                422
            );
        }

        $location = Location::create($validator->validated());

        return $this->success(
            $location,
            'Location created successfully',
            201
        );
    }

    /**
     * Display the specified location
     */
    public function show($id)
    {
        $location = Location::find($id);

        if (!$location) {
            return $this->error(
                [],
                'Location not found',
                404
            );
        }

        return $this->success(
            $location,
            'Location retrieved successfully'
        );
    }

    /**
     * Update the specified location
     */
    public function update(Request $request, $id)
    {
        $location = Location::find($id);

        if (!$location) {
            return $this->error(
                [],
                'Location not found',
                404
            );
        }

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'abbreviation' => 'nullable|string|max:50',
            'directions' => 'nullable|string',

            'print_card_line_1' => 'nullable|string|max:255',
            'print_card_line_2' => 'nullable|string|max:255',

            'address_1' => 'nullable|string|max:255',
            'address_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'zip' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:100',

            'contact_name' => 'nullable|string|max:255',
            'contact_email' => 'nullable|email|max:255',
            'contact_phone' => 'nullable|string|max:30',

            'internal_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return $this->error(
                $validator->errors(),
                'Validation failed',
                422
            );
        }

        $location->update($validator->validated());

        return $this->success(
            $location,
            'Location updated successfully'
        );
    }

    /**
     * Remove the specified location
     */
    public function destroy($id)
    {
        $location = Location::find($id);

        if (!$location) {
            return $this->error(
                [],
                'Location not found',
                404
            );
        }

        $location->delete();

        return $this->success(
            [],
            'Location deleted successfully'
        );
    }
}
