<?php

namespace App\Http\Controllers\API;

use App\Models\PromoCode;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;

class PromoCodeController extends Controller
{
    use ApiResponse;

    // Get All PromoCodes
    public function index()
    {
        $promoCodes = PromoCode::orderBy('id', 'asc')->get();
        // Load client for response
        $promoCodes->load(['client' => function($query) {
            $query->select('id', 'company');
        }]);

        return $this->success($promoCodes, "Data Retrieved Successfully", 200);
    }

    // Get Single PromoCode
    public function show(Request $request)
    {
        $promoCode = PromoCode::find($request->id);
        // Load client for response
        $promoCode->load(['client' => function($query) {
            $query->select('id', 'company');
        }]);
        return $this->success($promoCode, "Data Retrieved Successfully", 200);
    }

    // Create PromoCode
    public function store(Request $request)
    {
        $validated = $request->validate([
            'code'                         => 'required|string|max:50|unique:promo_code,code,' . ($promoCode->id ?? 'NULL'),
            'client_id'                    => 'nullable|exists:clients,id',
            'description'                  => 'nullable|string',
            'start_date'                   => 'required|date',
            'end_date'                     => 'required|date|after_or_equal:start_date',
            'type'                         => 'required|in:dollars_off,percentage_off',
            'discount'                     => 'required|numeric|min:0|max:999999.99',
            'max_uses'                     => 'nullable|integer|min:1',
            'apply_to_addons_and_shipping' => 'nullable|boolean',
            'restrict_by_course_type'      => 'nullable|boolean',
            'restrict_course_ids'          => 'required_if:restrict_by_course_type,1|array',
            'restrict_course_ids.*'        => 'exists:courses,id',
            'does_not_reduce_balance_due'  => 'nullable|boolean',
        ]);

        $promoCode = new PromoCode();
        $promoCode->code = $validated['code'];
        $promoCode->client_id = $validated['client_id'] ?? null;
        $promoCode->description = $validated['description'] ?? null;
        $promoCode->start_date = $validated['start_date'];
        $promoCode->end_date = $validated['end_date'];
        $promoCode->type = $validated['type'];
        $promoCode->discount = $validated['discount'];
        $promoCode->max_uses = $validated['max_uses'] ?? null;
        $promoCode->apply_to_addons_and_shipping = $validated['apply_to_addons_and_shipping'] ?? false;
        $promoCode->restrict_by_course_type = $validated['restrict_by_course_type'] ?? false;
        $promoCode->does_not_reduce_balance_due = $validated['does_not_reduce_balance_due'] ?? false;
        $promoCode->save();

        // Attach multiple restricted courses
        if (!empty($validated['restrict_course_ids'])) {
            $promoCode->restrictCourses()->sync($validated['restrict_course_ids']);
        }

        // Load client for response
        $promoCode->load(['client' => function($query) {
            $query->select('id', 'company');
        }]);

        return $this->success($promoCode, 'Data stored successfully!', 200);
    }

    // Update PromoCode
    public function update(Request $request)
    {
        $promoCode = PromoCode::find($request->id);

        if (!$promoCode) {
           return $this->error(null, 'Data not found', 404);
        }

        $validated = $request->validate([
            'code'                         => 'required|string',
            'client_id'                    => 'nullable|exists:clients,id',
            'description'                  => 'nullable|string',
            'start_date'                   => 'required|date',
            'end_date'                     => 'required|date|after_or_equal:start_date',
            'type'                         => 'required|in:dollars_off,percentage_off',
            'discount'                     => 'required|numeric|min:0|max:999999.99',
            'max_uses'                     => 'nullable|integer|min:1',
            'apply_to_addons_and_shipping' => 'nullable|boolean',
            'restrict_by_course_type'      => 'nullable|boolean',
            'restrict_course_ids'          => 'required_if:restrict_by_course_type,1|array',
            'restrict_course_ids.*'        => 'exists:courses,id',
            'does_not_reduce_balance_due'  => 'nullable|boolean',
        ]);

        $promoCode->code = $validated['code'];
        $promoCode->client_id = $validated['client_id'] ?? null;
        $promoCode->description = $validated['description'] ?? null;
        $promoCode->start_date = $validated['start_date'];
        $promoCode->end_date = $validated['end_date'];
        $promoCode->type = $validated['type'];
        $promoCode->discount = $validated['discount'];
        $promoCode->max_uses = $validated['max_uses'] ?? null;
        $promoCode->apply_to_addons_and_shipping = $validated['apply_to_addons_and_shipping'] ?? false;
        $promoCode->restrict_by_course_type = $validated['restrict_by_course_type'] ?? false;
        $promoCode->does_not_reduce_balance_due = $validated['does_not_reduce_balance_due'] ?? false;
        $promoCode->save();

        // Attach multiple restricted courses
        if (!empty($validated['restrict_course_ids'])) {
            $promoCode->restrictCourses()->sync($validated['restrict_course_ids']);
        }

        // Load client for response
        $promoCode->load(['client' => function($query) {
            $query->select('id', 'company');
        }]);

        return $this->success($promoCode, 'Data update successfully!', 200);
    }

    // Delete PromoCode
    public function destroy(Request $request)
    {
        $promoCode = PromoCode::find($request->id);

        if (!$promoCode) {
            return $this->error(null, 'Data not found', 404);
        }

        $promoCode->delete();
        return $this->success($promoCode, 'Data deleted successfully!', 200);
    }

}
