<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\TCProduct;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TCProductController extends Controller
{
    use ApiResponse;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $request->validate([
            'per_page' => 'nullable|integer|min:1',
            'search' => 'nullable|string',
        ]);
        $TCProducts = TCProduct::query()
            ->when($request->has('search'), function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                    ->orWhere('code', 'like', '%' . $request->search . '%');
            })
            ->orderBy('id', 'desc')->paginate($request->get('per_page', 10));

        if ($TCProducts->isEmpty()) {
            return $this->success([], 'No TC Products found', 404);
        }

        return $this->success($TCProducts, 'TC Products retrieved successfully', 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'code' => 'required|string|unique:t_c_products,code',
            'name' => 'required|string',
            'price_label' => 'required|numeric',
            'price' => 'required|numeric',
            'is_taxable' => 'nullable|numeric',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors(), $validator->errors()->first(), 422);
        }

        $TCProduct = TCProduct::create([
            'code' => $request->code,
            'name' => $request->name,
            'price_label' => $request->price_label,
            'price' => $request->price,
            'is_taxable' => $request->is_taxable ?? false,
        ]);

        return $this->success($TCProduct, 'TC Product created successfully', 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $TCProducts = TCProduct::where('id', $id)->first();

        if ($TCProducts == null) {
            return $this->success([], 'No TC Products found', 404);
        }

        return $this->success($TCProducts, 'TC Products retrieved successfully', 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $id)
    {
        $validator = Validator::make($request->all(), [
            'code' => 'required|string|unique:t_c_products,code,' . $id,
            'name' => 'required|string',
            'price_label' => 'required|numeric',
            'price' => 'required|numeric',
            'is_taxable' => 'nullable|numeric',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors(), $validator->errors()->first(), 422);
        }

        $TCProduct = TCProduct::where('id', $id)->first();

        if ($TCProduct == null) {
            return $this->success([], 'No TC Products found', 404);
        }

        $TCProduct->update([
            'code' => $request->code,
            'name' => $request->name,
            'price_label' => $request->price_label,
            'price' => $request->price,
            'is_taxable' => $request->is_taxable ?? false,
        ]);

        return $this->success($TCProduct, 'TC Product updated successfully', 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id)
    {
        $TCProduct = TCProduct::where('id', $id)->first();

        if ($TCProduct == null) {
            return $this->success([], 'No TC Products found', 404);
        }

        $TCProduct->delete();

        return $this->success([], 'TC Product deleted successfully', 200);
    }
}
