<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\TrainingSite;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TrainingSiteController extends Controller
{
    use ApiResponse;

    // Get Training Sites
    public function getTrainingSites()
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $getTrainingSites = TrainingSite::with('settings')->get();

        if ($getTrainingSites->isEmpty()) {
            return $this->error([], 'No Training Sites Found', 200);
        }

        return $this->success($getTrainingSites, 'Training Sites Retrieved Successfully', 200);
    }

    // Create Training Site
    public function getTrainingSiteCreate(Request $request)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $validator = Validator::make($request->all(), [
            'company_name'        => 'required|string|max:255',
            'training_center_name' => 'required|string|max:255',
            'contact_first_name'  => 'required|string|max:255',
            'contact_last_name'   => 'required|string|max:255',
            'email'               => 'required|email|unique:training_sites,email',
            'phone_number'        => 'required|string|max:50',
            'fax_number'          => 'nullable|string|max:50',
            'address_line_1'      => 'required|string|max:255',
            'address_line_2'      => 'nullable|string|max:255',
            'city'                => 'required|string|max:255',
            'state_province'      => 'required|string|max:255',
            'postal_code'         => 'required|string|max:50',
            'country'             => 'required|string|max:255',
            'training_site_id'    => 'nullable|string|max:255',
            'price_level'         => 'nullable|integer|min:0',
            'sales_tax_rate'      => 'nullable|integer|min:0',
            'notes'               => 'nullable|string',

            // Settings
            'enable_certification_card_printing'                      => 'nullable|boolean',
            'send_reminders_to_instructors_with_unfinalized_rosters'  => 'nullable|boolean',
            'create_an_admin_user_for_this_site'                      => 'nullable|boolean',
            'restrict_tc_product_orders_to_admins_only'               => 'nullable|boolean',
            'restrict_instructors_to_only_view_classes_they_teach'    => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors(), $validator->errors()->first(), 422);
        }
        try {
            $site = TrainingSite::create([
                'company_name'        => $request->company_name,
                'training_center_name' => $request->training_center_name,
                'contact_first_name'  => $request->contact_first_name,
                'contact_last_name'   => $request->contact_last_name,
                'email'               => $request->email,
                'phone_number'        => $request->phone_number,
                'fax_number'          => $request->fax_number,
                'address_line_1'      => $request->address_line_1,
                'address_line_2'      => $request->address_line_2,
                'city'                => $request->city,
                'state_province'      => $request->state_province,
                'postal_code'         => $request->postal_code,
                'country'             => $request->country,
                'training_site_id'    => $request->training_site_id,
                'price_level'         => $request->price_level ?? 0,
                'sales_tax_rate'      => $request->sales_tax_rate ?? 0,
                'notes'               => $request->notes
            ]);

            // Create Settings
            $settings = $site->settings()->create([
                'enable_certification_card_printing'                      => $request->enable_certification_card_printing ?? false,
                'send_reminders_to_instructors_with_unfinalized_rosters'  => $request->send_reminders_to_instructors_with_unfinalized_rosters ?? false,
                'create_an_admin_user_for_this_site'                      => $request->create_an_admin_user_for_this_site ?? false,
                'restrict_tc_product_orders_to_admins_only'               => $request->restrict_tc_product_orders_to_admins_only ?? false,
                'restrict_instructors_to_only_view_classes_they_teach'    => $request->restrict_instructors_to_only_view_classes_they_teach ?? false,
            ]);

            return $this->success([
                'training_site'  => $site,
                'settings'       => $settings
            ], "Training Site Created Successfully", 201);
        } catch (\Exception $e) {
            return $this->error([], $e->getMessage(), 500);
        }
    }
    public function editTrainingSite($id)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $site = TrainingSite::with('settings')->find($id);

        if (!$site) {
            return $this->error([], 'Training Site Not Found', 404);
        }

        return $this->success($site, 'Training Site Retrieved Successfully', 200);
    }
    public function updateTrainingSite(Request $request, $id)
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error([], 'Unauthorized', 401);
        }

        $site = TrainingSite::with('settings')->find($id);
        if (!$site) {
            return $this->error([], 'Training Site Not Found', 404);
        }

        $validator = Validator::make($request->all(), [
            'company_name'        => 'required|string|max:255',
            'training_center_name' => 'required|string|max:255',
            'contact_first_name'  => 'required|string|max:255',
            'contact_last_name'   => 'required|string|max:255',
            'email'               => 'required|email|unique:training_sites,email,' . $id,
            'phone_number'        => 'required|string|max:50',
            'fax_number'          => 'nullable|string|max:50',
            'address_line_1'      => 'required|string|max:255',
            'address_line_2'      => 'nullable|string|max:255',
            'city'                => 'required|string|max:255',
            'state_province'      => 'required|string|max:255',
            'postal_code'         => 'required|string|max:50',
            'country'             => 'required|string|max:255',
            'training_site_id'    => 'nullable|string|max:255',
            'price_level'         => 'nullable|integer|min:0',
            'sales_tax_rate'      => 'nullable|integer|min:0',
            'notes'               => 'nullable|string',

            // Settings
            'enable_certification_card_printing'                      => 'nullable|boolean',
            'send_reminders_to_instructors_with_unfinalized_rosters'  => 'nullable|boolean',
            'create_an_admin_user_for_this_site'                      => 'nullable|boolean',
            'restrict_tc_product_orders_to_admins_only'               => 'nullable|boolean',
            'restrict_instructors_to_only_view_classes_they_teach'    => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return $this->error($validator->errors(), $validator->errors()->first(), 422);
        }

        try {
            // Update Main Site
            $site->update([
                'company_name'        => $request->company_name,
                'training_center_name' => $request->training_center_name,
                'contact_first_name'  => $request->contact_first_name,
                'contact_last_name'   => $request->contact_last_name,
                'email'               => $request->email,
                'phone_number'        => $request->phone_number,
                'fax_number'          => $request->fax_number,
                'address_line_1'      => $request->address_line_1,
                'address_line_2'      => $request->address_line_2,
                'city'                => $request->city,
                'state_province'      => $request->state_province,
                'postal_code'         => $request->postal_code,
                'country'             => $request->country,
                'training_site_id'    => $request->training_site_id,
                'price_level'         => $request->price_level ?? 0,
                'sales_tax_rate'      => $request->sales_tax_rate ?? 0,
                'notes'               => $request->notes,
            ]);

            // Update Settings
            $site->settings()->update([
                'enable_certification_card_printing'                      => $request->enable_certification_card_printing ?? false,
                'send_reminders_to_instructors_with_unfinalized_rosters'  => $request->send_reminders_to_instructors_with_unfinalized_rosters ?? false,
                'create_an_admin_user_for_this_site'                      => $request->create_an_admin_user_for_this_site ?? false,
                'restrict_tc_product_orders_to_admins_only'               => $request->restrict_tc_product_orders_to_admins_only ?? false,
                'restrict_instructors_to_only_view_classes_they_teach'    => $request->restrict_instructors_to_only_view_classes_they_teach ?? false,
            ]);

            return $this->success([
                'training_site' => $site,
                'settings'      => $site->settings,
            ], "Training Site Updated Successfully", 200);

        } catch (\Exception $e) {
            return $this->error([], $e->getMessage(), 500);
        }
    }

}
